<?php
require_once __DIR__.'/config.php';

// -------- Telegram helpers --------
function tg_api($method, $params) {
    $url = "https://api.telegram.org/bot".BOT_TOKEN."/".$method;
    $opts = ['http'=>[
        'method'=>'POST',
        'header'=>"Content-Type: application/json\r\n",
        'content'=>json_encode($params)
    ]];
    return json_decode(file_get_contents($url, false, stream_context_create($opts)), true);
}
function tg_send($chat_id, $text, $reply_markup=null) {
    $p = ['chat_id'=>$chat_id,'text'=>$text,'parse_mode'=>'HTML'];
    if ($reply_markup) $p['reply_markup'] = $reply_markup;
    return tg_api('sendMessage', $p);
}
function tg_alert_admin($msg) {
    if (!ADMIN_TELEGRAM_ID) return;
    tg_send(ADMIN_TELEGRAM_ID, "⚠️ <b>System Alert</b>\n".$msg);
}

// -------- Bot state machine --------
function state_get($telegram_id) {
    $stmt = pdo()->prepare("SELECT state, temp_data FROM bot_states WHERE telegram_id=? LIMIT 1");
    $stmt->execute([$telegram_id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) return ['state'=>'idle','data'=>[]];
    return ['state'=>$row['state'], 'data'=>($row['temp_data']? json_decode($row['temp_data'], true):[])];
}
function state_set($telegram_id, $state, $data=[]) {
    $stmt = pdo()->prepare("INSERT INTO bot_states(telegram_id,state,temp_data) VALUES(?,?,?) ON DUPLICATE KEY UPDATE state=VALUES(state), temp_data=VALUES(temp_data)");
    $stmt->execute([$telegram_id, $state, json_encode($data)]);
}
function state_reset($telegram_id) {
    $stmt = pdo()->prepare("DELETE FROM bot_states WHERE telegram_id=?");
    $stmt->execute([$telegram_id]);
}

// -------- Users & balance --------
function user_get_or_create($telegram_id, $name) {
    $pdo = pdo();
    $pdo->prepare("INSERT IGNORE INTO users(telegram_id,name) VALUES(?,?)")->execute([$telegram_id,$name]);
    $stmt = $pdo->prepare("SELECT * FROM users WHERE telegram_id=? LIMIT 1");
    $stmt->execute([$telegram_id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}
function user_blocked($telegram_id) {
    $stmt = pdo()->prepare("SELECT status FROM users WHERE telegram_id=? LIMIT 1");
    $stmt->execute([$telegram_id]);
    $s = $stmt->fetchColumn();
    return ($s==='blocked');
}
function user_balance($telegram_id) {
    $stmt = pdo()->prepare("SELECT balance FROM users WHERE telegram_id=? LIMIT 1");
    $stmt->execute([$telegram_id]);
    return (float)$stmt->fetchColumn();
}
function user_add_balance($user_id, $amount, $method='Admin') {
    $pdo = pdo();
    $pdo->beginTransaction();
    $pdo->prepare("UPDATE users SET balance=balance+? WHERE id=?")->execute([$amount,$user_id]);
    $pdo->prepare("INSERT INTO topups(user_id, amount, method) VALUES(?,?,?)")->execute([$user_id,$amount,$method]);
    $pdo->commit();
}

// -------- APIs --------
function apis_enabled() {
    $stmt = pdo()->query("SELECT * FROM sms_apis WHERE enabled=1 ORDER BY id ASC");
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function api_get($id) {
    $stmt = pdo()->prepare("SELECT * FROM sms_apis WHERE id=? LIMIT 1");
    $stmt->execute([$id]);
    $api = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($api && (int)$api['enabled'] === 1) {
        return $api;
    }
    return null;
}

function api_check_balance($api) {
    if (empty($api['balance_endpoint'])) return ['ok'=>false,'balance'=>null,'raw'=>'No balance endpoint configured'];
    $payload = [
        'account'  => $api['username'],
        'password' => $api['password']
    ];
    $resp = http_json($api['balance_endpoint'], $payload);
    // Expect {balance: ...} or similar — store raw anyway
    $balance = null;
    if (isset($resp['json']['balance'])) $balance = $resp['json']['balance'];
    return ['ok'=>$resp['http']==200, 'balance'=>$balance, 'raw'=>$resp['raw']];
}

// -------- HTTP JSON --------
function http_json($url, $payload, $headers=['Content-Type: application/json']) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER=>true,
        CURLOPT_POST=>true,
        CURLOPT_HTTPHEADER=>$headers,
        CURLOPT_POSTFIELDS=>json_encode($payload),
        CURLOPT_TIMEOUT=>30
    ]);
    $raw = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    $json = json_decode($raw, true);
    return ['http'=>$http, 'raw'=>$raw, 'json'=>$json];
}

// -------- SMS sending & logs --------
function sms_cost_per_message() { return 0.33; } // adjust per your business

function sms_log($user_id,$api_id,$recipient,$message,$api_response,$status) {
    $stmt = pdo()->prepare("INSERT INTO sms_logs(user_id,api_id,recipient,message,api_response,status) VALUES(?,?,?,?,?,?)");
    $stmt->execute([$user_id,$api_id,$recipient,$message,$api_response,$status]);
}

function sms_send_via_api($user, $api_id, $numbers, $text) {
    $api = api_get($api_id);
    if (!$api || !$api['enabled']) return ['ok'=>false,'reason'=>'API disabled or not found'];

    // Build URL for GET request
    $url = $api['endpoint']."?account=".urlencode($api["username"]).
           "&password=".urlencode($api["password"]).
           "&smstype=0".
           "&numbers=".urlencode($numbers).
           "&content=".urlencode($text);

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30
    ]);
    $raw = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $json = json_decode($raw, true);

    // ✅ Consider success if API returns status=0 and success>=1
    $ok = ($http == 200 && isset($json['status']) && $json['status'] == 0 && ($json['success'] ?? 0) > 0);
    $status = $ok ? 'success' : 'failed';

    // Log
    sms_log($user['id'], $api['id'], $numbers, $text, $raw, $status);

    if (!$ok) {
        tg_alert_admin("⚠️ API <b>{$api['name']}</b> failed.\nHTTP: {$http}\nNumbers: <code>{$numbers}</code>\nRaw: <code>".htmlspecialchars(substr($raw,0,2000))."</code>");
    }

    return ['ok'=>$ok, 'http'=>$http, 'raw'=>$raw, 'json'=>$json];
}
