<?php
// admin/test_apis.php (updated - only selected APIs + confirmation modal)
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/auth.php';
require_login();
if (session_status() !== PHP_SESSION_ACTIVE) session_start();

include 'partials/header.php';

// fetch APIs
$rows = pdo()->query("SELECT * FROM sms_apis ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);
?>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-icons/1.11.3/font/bootstrap-icons.min.css">

<div class="d-flex align-items-center justify-content-between mb-3">
  <h3 class="mb-0">Test SMS APIs</h3>
  <small class="text-muted">Send a test message to a number via selected APIs</small>
</div>

<div class="card mb-4">
  <div class="card-body">
    <form id="testForm" class="row g-3">
      <div class="col-12 col-md-4">
        <label class="form-label small">Test Phone Number</label>
        <input type="tel" id="testNumber" class="form-control" placeholder="9876543210" required>
      </div>

      <div class="col-12 col-md-6">
        <label class="form-label small">Message (API name will be prepended)</label>
        <input type="text" id="testMessage" class="form-control" placeholder="Hello from MR-KIT" required>
      </div>

      <div class="col-12 col-md-2">
        <label class="form-label small">Which APIs?</label>
        <select id="whichApis" class="form-select">
          <option value="checked" selected>Checked only</option>
          <option value="active">Active only</option>
          <option value="inactive">Inactive only</option>
          <option value="both">Both</option>
        </select>
      </div>

      <div class="col-12 d-flex gap-2">
        <button id="runTest" type="button" class="btn btn-primary"><i class="bi bi-rocket"></i> Test Selected APIs</button>
        <button id="runAll" type="button" class="btn btn-outline-primary"><i class="bi bi-lightning"></i> Test All (Active)</button>
        <button id="clearResults" type="button" class="btn btn-outline-secondary">Clear Results</button>
      </div>
    </form>
  </div>
</div>

<!-- API list -->
<div class="card mb-4">
  <div class="card-body">
    <div class="row mb-2">
      <div class="col"><strong>APIs</strong></div>
      <div class="col-auto text-muted small">Toggle to include</div>
    </div>

    <div class="list-group" id="apiList" style="max-height:320px; overflow:auto;">
      <?php foreach($rows as $r):
        $enabled = (int)$r['enabled'];
        $badge = $enabled ? 'success' : 'secondary';
      ?>
        <label class="list-group-item d-flex justify-content-between align-items-start">
          <div class="ms-2 me-auto">
            <div class="fw-semibold"><?= htmlspecialchars($r['name']) ?> <small class="text-muted">#<?= (int)$r['id'] ?></small></div>
            <div class="small text-truncate" style="max-width:540px"><?= htmlspecialchars($r['endpoint']) ?></div>
            <div class="small text-muted">Balance endpoint: <?= htmlspecialchars($r['balance_endpoint'] ?? '—') ?></div>
          </div>
          <div class="d-flex flex-column align-items-end">
            <span class="badge bg-<?= $badge ?> mb-2"><?= $enabled ? 'Active' : 'Inactive' ?></span>
            <div>
              <!-- now checkboxes are real toggles for selection -->
              <input class="form-check-input api-checkbox" type="checkbox"
                     value="<?= (int)$r['id'] ?>"
                     data-name="<?= htmlspecialchars($r['name'], ENT_QUOTES) ?>"
                     data-enabled="<?= $enabled ? '1' : '0' ?>"
                     <?= $enabled ? 'checked' : '' ?>>
            </div>
          </div>
        </label>
      <?php endforeach; ?>
    </div>
  </div>
</div>

<!-- Results card (static) -->
<div class="card mb-4">
  <div class="card-body">
    <h5 class="card-title">Results (history)</h5>
    <div id="resultsArea" class="list-group"></div>
  </div>
</div>

<!-- Confirmation Modal -->
<div class="modal fade" id="confirmModal" tabindex="-1" aria-labelledby="confirmModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-md modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="confirmModalLabel"><i class="bi bi-exclamation-circle-fill text-warning"></i> Confirm Test</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <p id="confirmText" class="mb-1"></p>
        <p class="small text-muted">This will send the test message to the selected APIs. Proceed?</p>
      </div>
      <div class="modal-footer">
        <button id="confirmCancel" type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button id="confirmOk" type="button" class="btn btn-danger">Yes, send tests</button>
      </div>
    </div>
  </div>
</div>

<!-- Modal: Live Test Progress -->
<div class="modal fade" id="testModal" tabindex="-1" aria-labelledby="testModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="testModalLabel"><i class="bi bi-gear-wide-connected"></i> Testing APIs</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close" id="modalCloseBtn"></button>
      </div>
      <div class="modal-body">
        <div id="modalProgress" class="mb-3">
          <div class="d-flex align-items-center gap-3">
            <div id="modalSpinner" class="spinner-border text-primary" role="status" style="width:1.2rem;height:1.2rem"></div>
            <div><strong id="modalStatus">Queued...</strong> <small class="text-muted" id="modalSub">Preparing</small></div>
          </div>
        </div>

        <div class="table-responsive">
          <table class="table table-sm table-hover" id="modalTable">
            <thead class="table-light">
              <tr><th>API</th><th>Status</th><th>HTTP</th><th>Response (trim)</th></tr>
            </thead>
            <tbody id="modalBody">
              <!-- filled dynamically -->
            </tbody>
          </table>
        </div>
      </div>
      <div class="modal-footer">
        <button id="modalDone" type="button" class="btn btn-success" data-bs-dismiss="modal">Done</button>
      </div>
    </div>
  </div>
</div>

<!-- JS -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script>
$(function(){
  const testModalEl = document.getElementById('testModal');
  const bsModal = new bootstrap.Modal(testModalEl, { backdrop: 'static', keyboard: false });
  const confirmModalEl = document.getElementById('confirmModal');
  const bsConfirm = new bootstrap.Modal(confirmModalEl, { backdrop: 'static', keyboard: false });

  function appendResultHistory(apiName, apiId, html) {
    $('#resultsArea').prepend(
      $('<div>').addClass('list-group-item')
        .append($('<div>').addClass('d-flex w-100 justify-content-between')
          .append($('<h6>').addClass('mb-1').text(apiName + ' (ID: ' + apiId + ')'))
          .append($('<small>').addClass('text-muted').text(new Date().toLocaleString()))
        )
        .append($('<p>').addClass('mb-1').html(html))
    );
  }

  // Only include APIs that are CHECKED. If which==='active'/'inactive'/'both', filter by data-enabled too.
  function getSelectedApis(filter) {
    const arr = [];
    $('.api-checkbox:checked').each(function(){
      const enabled = $(this).data('enabled') == '1';
      if (filter === 'active' && !enabled) return;
      if (filter === 'inactive' && enabled) return;
      arr.push({ id: $(this).val(), name: $(this).data('name') });
    });
    return arr;
  }

  function escapeHtml(s){ return String(s).replaceAll('&','&amp;').replaceAll('<','&lt;').replaceAll('>','&gt;'); }

  // show confirmation modal before running tests
  let pendingRun = null; // {apis, number, message}
  function showConfirm(apis, number, message) {
    const count = apis.length;
    const names = apis.map(a => escapeHtml(a.name)).slice(0,6).join(', ');
    $('#confirmText').html(`<strong>${count}</strong> API(s) will be tested.<br><strong>Number:</strong> ${escapeHtml(number)}<br><strong>Message:</strong> ${escapeHtml(message)}<br><small class="text-muted">APIs: ${names}${apis.length>6 ? '…' : ''}</small>`);
    pendingRun = { apis, number, message };
    bsConfirm.show();
  }

  // After confirming, actually run tests
  $('#confirmOk').click(function(){
    bsConfirm.hide();
    if (!pendingRun) return;
    const { apis, number, message } = pendingRun;
    pendingRun = null;
    runTests(apis, number, message);
  });

  function runTests(apiList, number, message) {
    if (!apiList.length) {
      alert('No APIs selected'); return;
    }

    // show modal and reset table
    $('#modalBody').empty();
    $('#modalStatus').text('Starting...');
    $('#modalSub').text('Sending test messages');
    $('#modalSpinner').show();
    bsModal.show();

    let i = 0;
    function next(){
      if (i >= apiList.length) {
        $('#modalStatus').text('Completed');
        $('#modalSub').text('All tests finished');
        $('#modalSpinner').hide();
        return;
      }
      const api = apiList[i++];
      // add a row for this API in modal table
      const rowId = 'row-' + api.id + '-' + Math.random().toString(36).slice(2,8);
      const $row = $('<tr>').attr('id', rowId)
        .append($('<td>').text(api.name + ' (ID:' + api.id + ')'))
        .append($('<td>').html('<span class="text-info">Sending...</span>'))
        .append($('<td>').text('—'))
        .append($('<td>').text('—'));
      $('#modalBody').append($row);
      $('#modalStatus').text('Testing: ' + api.name);

      // Perform AJAX to send_test.php
      $.ajax({
        url: 'send_test.php',
        method: 'POST',
        dataType: 'json',
        data: { id: api.id, to: number, message: message },
        timeout: 30000,
        success: function(res){
          let httpCode = res.http_code || (res.http || '—');
          let respSample = res.response_sample || (res.resp || '') || '';
          if (res.error) {
            $row.find('td').eq(1).html('<span class="text-danger">Error</span>');
            $row.find('td').eq(2).text(res.http_code || '-');
            $row.find('td').eq(3).html('<pre class="mb-0 small text-break text-danger">' + escapeHtml(res.error) + '</pre>');
            appendResultHistory(api.name, api.id, '<span class="text-danger">Error: ' + escapeHtml(res.error) + '</span>');
          } else {
            $row.find('td').eq(1).html('<span class="text-success">Done</span>');
            $row.find('td').eq(2).text(httpCode);
            $row.find('td').eq(3).html('<pre class="mb-0 small text-break">' + escapeHtml(String(respSample).substring(0,400)) + '</pre>');
            appendResultHistory(api.name, api.id, '<div><strong>HTTP:</strong> ' + escapeHtml(httpCode) + '</div><div><strong>Resp:</strong> ' + escapeHtml(String(respSample).substring(0,400)) + '</div>');
          }
          setTimeout(next, 350);
        },
        error: function(xhr, status, err) {
          let body = xhr.responseText || '';
          try {
            const j = JSON.parse(body);
            if (j.error) body = j.error;
          } catch(e){}
          $row.find('td').eq(1).html('<span class="text-danger">Failed</span>');
          $row.find('td').eq(2).text(xhr.status || '-');
          $row.find('td').eq(3).html('<pre class="mb-0 small text-break text-danger">' + escapeHtml(body).substring(0,400) + '</pre>');
          appendResultHistory(api.name, api.id, '<span class="text-danger">Request failed: ' + escapeHtml(body) + '</span>');
          setTimeout(next, 350);
        }
      });
    }

    next();
  }

  // Click handlers: show confirmation first
  $('#runTest').click(function(){
    const number = $('#testNumber').val().trim();
    const message = $('#testMessage').val().trim();
    if (!number || !message) { alert('Enter number and message'); return; }
    const which = $('#whichApis').val();
    const apis = getSelectedApis(which);
    if (!apis.length) { alert('No APIs selected'); return; }
    showConfirm(apis, number, message);
  });

  $('#runAll').click(function(){
    const number = $('#testNumber').val().trim();
    const message = $('#testMessage').val().trim();
    if (!number || !message) { alert('Enter number and message'); return; }
    // select all currently checked that are active
    const apis = [];
    $('.api-checkbox:checked').each(function(){
      if ($(this).data('enabled') == '1') apis.push({ id: $(this).val(), name: $(this).data('name') });
    });
    if (!apis.length) { alert('No active APIs selected'); return; }
    showConfirm(apis, number, message);
  });

  $('#clearResults').click(function(){ $('#resultsArea').empty(); });

});
</script>

<?php include 'partials/footer.php'; ?>
