<?php
// admin/apis.php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/auth.php';
require_login();
if (session_status() !== PHP_SESSION_ACTIVE) session_start();

// handle AJAX toggle first (returns JSON when requested)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['toggle'])) {
    $id = (int)($_POST['id'] ?? 0);
    $stmt = pdo()->prepare("SELECT enabled FROM sms_apis WHERE id = ?");
    $stmt->execute([$id]);
    $cur = $stmt->fetchColumn();
    if ($cur === false) {
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH'])) {
            header('Content-Type: application/json');
            echo json_encode(['error' => 'Not found']);
            exit;
        } else {
            $_SESSION['flash'] = ['type' => 'danger', 'msg' => 'API not found'];
            header('Location: ' . basename(__FILE__));
            exit;
        }
    }
    $new = $cur ? 0 : 1;
    $u = pdo()->prepare("UPDATE sms_apis SET enabled = ? WHERE id = ?");
    $u->execute([$new, $id]);

    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH'])) {
        header('Content-Type: application/json');
        echo json_encode(['ok' => true, 'enabled' => (int)$new]);
        exit;
    } else {
        $_SESSION['flash'] = ['type' => 'success', 'msg' => 'API toggled'];
        header('Location: ' . basename(__FILE__));
        exit;
    }
}

// POST handling (add / edit standard flow)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add'])) {
        try {
            $stmt = pdo()->prepare("INSERT INTO sms_apis(name,endpoint,balance_endpoint,username,password,enabled) VALUES(?,?,?,?,?,?)");
            $stmt->execute([
                trim($_POST['name']),
                trim($_POST['endpoint']),
                trim($_POST['balance_endpoint']) ?: null,
                trim($_POST['username']),
                trim($_POST['password']),
                (int)!empty($_POST['enabled'])
            ]);
            $_SESSION['flash'] = ['type' => 'success', 'msg' => 'API added'];
        } catch (\PDOException $ex) {
            error_log("Failed to add API: " . $ex->getMessage());
            $_SESSION['flash'] = ['type' => 'danger', 'msg' => 'Failed to add API — check server logs'];
        }
        header('Location: ' . basename(__FILE__));
        exit;
    }

    if (isset($_POST['edit'])) {
        $stmt = pdo()->prepare("UPDATE sms_apis SET name=?, endpoint=?, balance_endpoint=?, username=?, password=?, enabled=? WHERE id=?");
        $stmt->execute([
            trim($_POST['name']),
            trim($_POST['endpoint']),
            trim($_POST['balance_endpoint']) ?: null,
            trim($_POST['username']),
            trim($_POST['password']),
            (int)!empty($_POST['enabled']),
            (int)$_POST['id']
        ]);
        $_SESSION['flash'] = ['type' => 'success', 'msg' => 'API updated'];
        header('Location: ' . basename(__FILE__));
        exit;
    }
}

include 'partials/header.php';

// Fetch APIs
$rows = pdo()->query("SELECT * FROM sms_apis ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);

// Prepare 7-day usage per API (server-side aggregation)
$usage_map = [];
$days = [];
$dt = new DateTimeImmutable('today', new DateTimeZone(date_default_timezone_get()));
for ($i = 6; $i >= 0; $i--) {
    $d = $dt->sub(new DateInterval("P{$i}D"))->format('Y-m-d');
    $days[] = $d;
}
foreach ($rows as $r) {
    $id = (int)$r['id'];
    $counts = array_fill(0, 7, 0);
    $start_date = $days[0] . ' 00:00:00';
    $stmt = pdo()->prepare("SELECT DATE(created_at) as d, COUNT(*) as c FROM sms_logs WHERE api_id = ? AND created_at >= ? GROUP BY DATE(created_at)");
    $stmt->execute([$id, $start_date]);
    $res = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $map = [];
    foreach ($res as $rr) $map[$rr['d']] = (int)$rr['c'];
    foreach ($days as $idx => $day) {
        $counts[$idx] = $map[$day] ?? 0;
    }
    $usage_map[$id] = $counts;
}

// flash
$flash = $_SESSION['flash'] ?? null;
unset($_SESSION['flash']);

function esc($v){ return htmlspecialchars($v ?? '', ENT_QUOTES); }
?>

<!-- Page-specific styles (futuristic theme + iOS toggle) -->
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css">
<link rel="stylesheet" href="https://cdn.datatables.net/responsive/2.5.0/css/responsive.bootstrap5.min.css">

<style>
:root{
  --accent-1: #7c5cff;
  --accent-2: #00e0b8;
  --muted: rgba(255,255,255,0.66);
  --card-radius: 14px;
  --ios-track-off: rgba(255,255,255,0.06);
  --ios-track-on: linear-gradient(90deg,var(--accent-1),var(--accent-2));
}

/* api-card */
.api-card {
  border-radius: var(--card-radius);
  background: linear-gradient(180deg, rgba(255,255,255,0.02), rgba(255,255,255,0.01));
  border: 1px solid rgba(255,255,255,0.05);
  box-shadow: 0 12px 40px rgba(2,6,23,0.5);
  padding: 14px;
  margin-bottom:12px;
  display:flex;
  gap:12px;
  align-items:center;
  transition: transform .12s ease, box-shadow .12s ease;
}
.api-card:hover { transform: translateY(-6px); box-shadow: 0 20px 56px rgba(2,6,23,0.6); }

.api-left { width:220px; min-width:160px; display:flex; flex-direction:column; gap:6px; }
.api-id { color: rgba(255,255,255,0.6); font-size:12px; }
.api-name { font-weight:800; font-size:16px; }

.api-endpoint { flex:1; color: rgba(255,255,255,0.9); font-size:13px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap; }

.api-mid { width:260px; display:flex; flex-direction:column; gap:6px; }
.sparkline { width:140px; height:34px; display:inline-block; vertical-align:middle; }

.balance-box {
  border-radius:10px;
  padding:8px 10px;
  min-width:150px;
  text-align:center;
  font-weight:700;
  background: linear-gradient(180deg, rgba(255,255,255,0.01), rgba(255,255,255,0.02));
  border:1px solid rgba(255,255,255,0.03);
}
.balance-neutral { color: var(--muted); }
.balance-green { color: #12a452; }
.balance-orange { color: #e08a00; }
.balance-red { color: #e03b3b; }

.api-actions { display:flex; gap:8px; align-items:center; min-width:280px; justify-content:flex-end; }

/* neon buttons */
.btn-neon {
  background: linear-gradient(90deg,var(--accent-1),var(--accent-2));
  color:#02102a; border-radius:10px; border:none; padding:8px 12px; font-weight:700;
  box-shadow: 0 12px 36px rgba(124,92,255,0.16);
}
.btn-outline-glass {
  background: transparent; border:1px solid rgba(255,255,255,0.06); color: rgba(255,255,255,0.9); border-radius:10px; padding:7px 10px;
}
.btn-ghost {
  background: linear-gradient(90deg, rgba(124,92,255,0.06), rgba(0,224,184,0.03)); color: #eaf2ff; border-radius:10px; padding:7px 10px; border: none;
}

/* iOS style toggle (track + thumb) */
.ios-toggle {
  --track-w: 52px;
  --track-h: 32px;
  display:inline-block;
  width:var(--track-w);
  height:var(--track-h);
  border-radius:999px;
  position:relative;
  box-sizing:content-box;
  cursor:pointer;
  transition: background .18s ease, box-shadow .18s ease;
  background: var(--ios-track-off);
  border: 1px solid rgba(255,255,255,0.05);
  padding:4px;
}
.ios-toggle .ios-thumb {
  width:24px;
  height:24px;
  border-radius:50%;
  background: #ffffff;
  position:absolute;
  top:50%;
  left:4px;
  transform: translateY(-50%);
  transition: left .18s cubic-bezier(.2,.9,.2,1), box-shadow .18s ease, transform .18s ease;
  box-shadow: 0 6px 16px rgba(2,6,23,0.35);
}

/* active state */
.ios-toggle.active {
  background: var(--ios-track-on);
  border-color: rgba(124,92,255,0.22);
  box-shadow: 0 8px 30px rgba(124,92,255,0.08);
}
.ios-toggle.active .ios-thumb {
  left: calc(var(--track-w) - 28px); /* align to right inner */
  box-shadow: 0 6px 22px rgba(124,92,255,0.22), 0 0 10px rgba(0,224,184,0.18);
  transform: translateY(-50%) scale(1.02);
  /* neon glow */
  filter: drop-shadow(0 0 6px rgba(124,92,255,0.8)) drop-shadow(0 0 14px rgba(0,224,184,0.35));
  animation: iosPulse 1.6s infinite;
}

/* subtle pulse */
@keyframes iosPulse {
  0% { transform: translateY(-50%) scale(1); filter: drop-shadow(0 0 4px rgba(124,92,255,0.6)); }
  50% { transform: translateY(-51%) scale(1.03); filter: drop-shadow(0 0 12px rgba(124,92,255,0.95)); }
  100% { transform: translateY(-50%) scale(1); filter: drop-shadow(0 0 4px rgba(124,92,255,0.6)); }
}

/* small-muted */
.small-muted { color: var(--muted); font-size:13px; }

@media (max-width:980px){
  .api-card{ flex-direction:column; align-items:stretch; }
  .api-actions { justify-content:space-between; width:100%; }
  .api-mid { width:100%; display:flex; justify-content:space-between; align-items:center; }
}
</style>

<div class="d-flex align-items-center justify-content-between mb-3">
  <div>
    <div style="font-weight:800;font-size:20px">SMS APIs</div>
    <div class="small-muted">Futuristic view — iOS-style toggles (confirm before disabling)</div>
  </div>
  <div>
    <button id="openAddApiBtn" class="btn-neon" type="button"><i class="bi bi-plus-lg me-1"></i> New API</button>
  </div>
</div>

<?php if ($flash): ?>
  <div class="api-card mb-3" style="align-items:center;justify-content:space-between;">
    <div style="font-weight:700"><?= esc($flash['type'] === 'success' ? 'Success' : 'Notice') ?></div>
    <div class="small-muted"><?= esc($flash['msg']) ?></div>
  </div>
<?php endif; ?>

<!-- Add API Modal -->
<div class="modal fade" id="addApiModal" tabindex="-1" aria-labelledby="addApiModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content">
      <form method="post" id="addApiForm" class="needs-validation" novalidate>
        <input type="hidden" name="add" value="1">

        <div class="modal-header">
          <h5 class="modal-title" id="addApiModalLabel">Add new SMS API</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>

        <div class="modal-body">
          <div class="row g-3">
            <div class="col-12 col-md-6">
              <label class="form-label small-muted">Name</label>
              <input class="form-control" name="name" placeholder="e.g. Twilio" required>
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label small-muted">Username</label>
              <input class="form-control" name="username" placeholder="account" required>
            </div>

            <div class="col-12">
              <label class="form-label small-muted">Endpoint</label>
              <input class="form-control" type="url" name="endpoint" placeholder="https://api.example.com/send" required>
            </div>

            <div class="col-12">
              <label class="form-label small-muted">Balance Endpoint (optional)</label>
              <input class="form-control" type="url" name="balance_endpoint" placeholder="https://api.example.com/balance">
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label small-muted">Password / API Key</label>
              <input class="form-control" name="password" placeholder="secret" required>
            </div>

            <div class="col-12 col-md-6 d-flex align-items-center">
              <div>
                <div class="small-muted" style="margin-bottom:6px">Enabled</div>
                <label class="ios-toggle" id="addModalIosToggle" tabindex="0" role="switch" aria-checked="true">
                  <span class="ios-thumb"></span>
                </label>
                <input type="hidden" id="addModalEnabledHidden" name="enabled" value="1">
              </div>
            </div>
          </div>
        </div>

        <div class="modal-footer">
          <button type="button" class="btn btn-outline-glass" data-bs-dismiss="modal">Cancel</button>
          <button type="submit" id="addApiSubmitBtn" class="btn btn-neon">
            <span id="addApiSubmitText">Add API</span>
            <span id="addApiSpinner" class="spinner-border spinner-border-sm ms-2 d-none" role="status" aria-hidden="true"></span>
          </button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- confirmation modal for disabling -->
<div class="modal fade" id="confirmDisableModal" tabindex="-1" aria-labelledby="confirmDisableLabel" aria-hidden="true">
  <div class="modal-dialog modal-sm modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="confirmDisableLabel">Confirm disable</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <p class="small-muted">Are you sure you want to disable this API? Disabled APIs will stop sending messages.</p>
        <input type="hidden" id="confirm-disable-id" value="">
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-outline-glass" data-bs-dismiss="modal">Cancel</button>
        <button type="button" id="confirmDisableBtn" class="btn btn-neon">Disable API</button>
      </div>
    </div>
  </div>
</div>

<!-- cards list -->
<?php foreach ($rows as $r):
    $id = (int)$r['id'];
    $usage = $usage_map[$id] ?? array_fill(0,7,0);
    $usage_json = esc(json_encode(array_values($usage)));
    $enabled = (bool)$r['enabled'];
?>
  <div class="api-card" id="api-card-<?= $id ?>" data-id="<?= $id ?>" data-usage='<?= $usage_json ?>'>
    <div class="api-left">
      <div class="api-id">#<?= $id ?></div>
      <div class="api-name"><?= esc($r['name']) ?> <?php if (!$enabled):?><span style="opacity:.6;font-weight:600;margin-left:6px;font-size:12px;color:#ff7b7b">inactive</span><?php endif;?></div>
      <div class="small-muted"><?= esc($r['username']) ?></div>
    </div>

    <div class="api-endpoint" title="<?= esc($r['endpoint']) ?>"><?= esc($r['endpoint']) ?></div>

    <div class="api-mid">
      <div class="balance-box" id="balance-<?= $id ?>">
        <span class="balance-neutral">Checking...</span>
      </div>

      <div>
        <div class="small-muted" style="margin-bottom:6px">7-day</div>
        <div class="sparkline" id="spark-<?= $id ?>" aria-hidden="true"></div>
      </div>
    </div>

    <div class="api-actions">
      <button class="btn-ghost" title="View logs" onclick="location.href='logs.php?api_id=<?= $id ?>'"><i class="bi bi-list-check" style="margin-right:6px"></i>Logs</button>

      <button class="btn-outline-glass edit-btn"
              data-id="<?= $id ?>"
              data-name="<?= esc($r['name']) ?>"
              data-endpoint="<?= esc($r['endpoint']) ?>"
              data-balance="<?= esc($r['balance_endpoint']) ?>"
              data-username="<?= esc($r['username']) ?>"
              data-password="<?= esc($r['password']) ?>"
              data-enabled="<?= $enabled ? '1' : '0' ?>">
        <i class="bi bi-pencil"></i> Edit
      </button>

      <!-- ios-style toggle control -->
      <label class="ios-toggle toggle" data-id="<?= $id ?>" role="switch" tabindex="0" aria-checked="<?= $enabled ? 'true' : 'false' ?>">
        <span class="ios-thumb"></span>
      </label>

      <button class="btn-neon" data-id="<?= $id ?>" onclick="fetchBalanceForRow(<?= $id ?>)" title="Refresh balance"><i class="bi bi-arrow-repeat" style="margin-right:8px"></i>Refresh</button>
    </div>
  </div>
<?php endforeach; ?>

<!-- Edit Modal (toggle inside modal) -->
<div class="modal fade" id="editModal" tabindex="-1" aria-labelledby="editModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content">
      <form method="post" id="editForm">
        <input type="hidden" name="edit" value="1">
        <input type="hidden" name="id" id="apiId">

        <div class="modal-header">
          <h5 class="modal-title" id="editModalLabel">Edit API</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
        </div>

        <div class="modal-body">
          <div class="row g-3">
            <div class="col-12 col-md-6">
              <label class="form-label small-muted">Name</label>
              <input type="text" class="form-control" name="name" id="apiName" required>
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label small-muted">Username</label>
              <input type="text" class="form-control" name="username" id="apiUser" required>
            </div>

            <div class="col-12">
              <label class="form-label small-muted">Endpoint</label>
              <input type="url" class="form-control" name="endpoint" id="apiEndpoint" required>
            </div>

            <div class="col-12">
              <label class="form-label small-muted">Balance Endpoint (optional)</label>
              <input type="url" class="form-control" name="balance_endpoint" id="apiBalance">
            </div>

            <div class="col-12 col-md-6">
              <label class="form-label small-muted">Password / API Key</label>
              <input type="text" class="form-control" name="password" id="apiPassword" required>
            </div>

            <div class="col-12 col-md-6 d-flex align-items-center justify-content-between">
              <!-- Visual toggle in modal (iOS style) -->
              <div style="display:flex;flex-direction:column;">
                <div class="small-muted" style="margin-bottom:6px">Enabled</div>
                <label class="ios-toggle" id="modalIosToggle" tabindex="0" role="switch">
                  <span class="ios-thumb"></span>
                </label>
                <input type="hidden" name="enabled" id="modalEnabledHidden" value="0">
              </div>

              <div>
                <div class="ms-3">
                  <button type="button" class="btn btn-sm btn-outline-glass" id="revealPassword"><i class="bi bi-eye"></i> Reveal</button>
                </div>
              </div>
            </div>
          </div> <!-- /.row -->
        </div>

        <div class="modal-footer">
          <button type="button" class="btn btn-outline-glass" data-bs-dismiss="modal">Cancel</button>
          <button type="submit" class="btn btn-neon"><i class="bi bi-save me-1"></i> Save changes</button>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- Toast -->
<div class="position-fixed bottom-0 end-0 p-3" style="z-index: 1080">
  <div id="genericToast" class="toast align-items-center text-bg-info border-0" role="alert" aria-live="polite" aria-atomic="true">
    <div class="d-flex">
      <div class="toast-body" id="genericToastBody">Message</div>
      <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
    </div>
  </div>
</div>

<!-- Scripts -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/dataTables.responsive.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/responsive.bootstrap5.min.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function(){

  // DataTable initialization if you keep a hidden table for data (no harm)
  try { $('#apisTable').DataTable(); } catch(e){}

  // sparkline renderer
  function drawSparkline(containerId, data){
    const el = document.getElementById(containerId);
    if (!el) return;
    const w = el.clientWidth || 140;
    const h = el.clientHeight || 34;
    const max = Math.max(...data, 1);
    const min = Math.min(...data);
    const stepX = w / (data.length - 1 || 1);
    let path = '';
    data.forEach((v, i) => {
      const x = Math.round(i * stepX);
      const y = h - Math.round(((v - min) / (max - min || 1)) * (h - 6)) - 3;
      path += (i === 0 ? `M ${x} ${y}` : ` L ${x} ${y}`);
    });
    const svg = `<svg viewBox="0 0 ${w} ${h}" width="${w}" height="${h}" xmlns="http://www.w3.org/2000/svg" preserveAspectRatio="none">
      <defs>
        <linearGradient id="g${containerId}" x1="0" x2="0" y1="0" y2="1">
          <stop offset="0%" stop-color="#7c5cff" stop-opacity="0.45"/>
          <stop offset="100%" stop-color="#00e0b8" stop-opacity="0.05"/>
        </linearGradient>
      </defs>
      <path d="${path}" fill="none" stroke="#7c5cff" stroke-width="1.6" stroke-linejoin="round" stroke-linecap="round"/>
      <path d="${path} L ${w} ${h} L 0 ${h} Z" fill="url(#g${containerId})" opacity="0.7"/>
    </svg>`;
    el.innerHTML = svg;
  }

  // populate sparklines
  document.querySelectorAll('.api-card').forEach(card=>{
    const id = card.dataset.id;
    let usage = [];
    try { usage = JSON.parse(card.getAttribute('data-usage') || '[]'); } catch(e){}
    if (usage && usage.length) drawSparkline('spark-' + id, usage);
  });

  // Add API modal wiring
  const addModalEl = document.getElementById('addApiModal');
  const addModal = new bootstrap.Modal(addModalEl);
  document.getElementById('openAddApiBtn')?.addEventListener('click', ()=> addModal.show());

  // Add modal iOS toggle wiring
  const addIosToggle = document.getElementById('addModalIosToggle');
  if (addIosToggle) {
    // default active
    addIosToggle.classList.add('active');
    addIosToggle.querySelector('.ios-thumb').style.left = 'calc(var(--track-w) - 28px)';
    document.getElementById('addModalEnabledHidden').value = '1';
    addIosToggle.addEventListener('click', function(){
      this.classList.toggle('active');
      const enabled = this.classList.contains('active');
      if (enabled) {
        this.querySelector('.ios-thumb').style.left = 'calc(var(--track-w) - 28px)';
        document.getElementById('addModalEnabledHidden').value = '1';
      } else {
        this.querySelector('.ios-thumb').style.left = '4px';
        document.getElementById('addModalEnabledHidden').value = '0';
      }
    });
    addIosToggle.addEventListener('keydown', function(e){ if (e.key===' '||e.key==='Enter'){ e.preventDefault(); this.click(); }});
  }

  // Add form UX: spinner + disable
  const addForm = document.getElementById('addApiForm');
  addForm?.addEventListener('submit', function(){
    const btn = document.getElementById('addApiSubmitBtn'), spinner = document.getElementById('addApiSpinner');
    if (btn) { btn.setAttribute('disabled','disabled'); if (spinner) spinner.classList.remove('d-none'); }
    // allow native submit -> page will reload on success/failure and show flash
  });

  // Edit modal logic
  let editModal = new bootstrap.Modal(document.getElementById('editModal'), {});
  $(document).on('click', '.edit-btn', function(){
    const btn = $(this);
    $('#apiId').val(btn.data('id'));
    $('#apiName').val(btn.data('name'));
    $('#apiEndpoint').val(btn.data('endpoint'));
    $('#apiBalance').val(btn.data('balance'));
    $('#apiUser').val(btn.data('username'));
    $('#apiPassword').val(btn.data('password'));

    // modal ios toggle sync
    const enabled = btn.data('enabled') == '1';
    const modalToggle = document.getElementById('modalIosToggle');
    const modalThumb = modalToggle.querySelector('.ios-thumb');
    if (enabled) {
      modalToggle.classList.add('active');
      modalThumb.style.left = 'calc(var(--track-w) - 28px)';
      document.getElementById('modalEnabledHidden').value = '1';
    } else {
      modalToggle.classList.remove('active');
      modalThumb.style.left = '4px';
      document.getElementById('modalEnabledHidden').value = '0';
    }

    editModal.show();
  });

  // modal toggle
  const modalToggleEl = document.getElementById('modalIosToggle');
  modalToggleEl?.addEventListener('click', function(){
    this.classList.toggle('active');
    const thumb = this.querySelector('.ios-thumb');
    if (this.classList.contains('active')) { thumb.style.left = 'calc(var(--track-w) - 28px)'; document.getElementById('modalEnabledHidden').value = '1'; }
    else { thumb.style.left = '4px'; document.getElementById('modalEnabledHidden').value = '0'; }
  });
  modalToggleEl?.addEventListener('keydown', function(e){ if (e.key===' '||e.key==='Enter'){ e.preventDefault(); this.click(); }});

  // reveal password in modal
  $('#revealPassword').on('click', function(){
    const pwd = $('#apiPassword');
    if (pwd.attr('type') === 'password') { pwd.attr('type','text'); $(this).html('<i class="bi bi-eye-slash"></i> Hide'); }
    else { pwd.attr('type','password'); $(this).html('<i class="bi bi-eye"></i> Reveal'); }
  });

  // Balance fetch function (exposed)
  window.fetchBalanceForRow = function(id){
    const target = document.getElementById('balance-' + id);
    if (!target) return;
    target.innerHTML = '<span style="opacity:0.85"><span class="spinner-border spinner-border-sm me-1" role="status" aria-hidden="true"></span>Checking...</span>';
    $.getJSON('check_balance.php', { id: id }, function(res){
      if (res.error) {
        target.innerHTML = '<span class="small" style="color:#ff6b6b">Error</span>';
        target.className = 'balance-box balance-neutral';
      } else {
        let val = (typeof res.balance === 'string') ? res.balance : (res.balance===null ? '' : String(res.balance));
        val = String(val).trim();
        let num = null;
        const m = val.match(/-?\d+(\.\d+)?/);
        if (m) num = parseFloat(m[0]);
        target.className = 'balance-box';
        if (num !== null && !isNaN(num)) {
          if (num < 10) target.classList.add('balance-red');
          else if (num < 50) target.classList.add('balance-orange');
          else target.classList.add('balance-green');
        } else {
          target.classList.add('balance-neutral');
        }
        let disp = val.length > 80 ? val.substring(0,80) + '…' : val;
        target.innerHTML = escHtml(disp);
      }
    }).fail(function(){
      target.innerHTML = '<span class="small" style="color:#ff6b6b">Fetch failed</span>';
      target.className = 'balance-box balance-neutral';
    });
  };

  // initial balance checks spaced
  document.querySelectorAll('.api-card').forEach((card, i)=>{
    const id = card.dataset.id;
    setTimeout(()=> fetchBalanceForRow(id), i * 240);
  });

  // Toggle handlers with confirmation on disable
  const confirmModalEl = document.getElementById('confirmDisableModal');
  const confirmModal = new bootstrap.Modal(confirmModalEl);
  let pendingToggle = null;

  document.querySelectorAll('.ios-toggle.toggle').forEach(t=>{
    const id = t.dataset.id;
    // initialize per server value (aria-checked attr)
    const activeInit = t.getAttribute('aria-checked') === 'true' || t.classList.contains('active');
    if (activeInit) {
      t.classList.add('active');
      t.querySelector('.ios-thumb').style.left = 'calc(var(--track-w) - 28px)';
    } else {
      t.classList.remove('active');
      t.querySelector('.ios-thumb').style.left = '4px';
    }

    t.addEventListener('click', function(e){
      // if currently active -> will disable -> ask confirmation
      const willDisable = this.classList.contains('active');
      if (willDisable) {
        pendingToggle = { el: this, id: id };
        document.getElementById('confirm-disable-id').value = id;
        confirmModal.show();
        return;
      }
      // enabling -> do request immediately
      doToggleRequest(this, id);
    });

    t.addEventListener('keydown', function(e){ if (e.key===' '||e.key==='Enter'){ e.preventDefault(); this.click(); }});
  });

  // confirm disable proceed
  document.getElementById('confirmDisableBtn').addEventListener('click', function(){
    const pid = document.getElementById('confirm-disable-id').value;
    if (!pendingToggle || pendingToggle.id != pid) {
      const el = document.querySelector(`.ios-toggle.toggle[data-id="${pid}"]`);
      if (el) doToggleRequest(el, pid);
      confirmModal.hide();
      return;
    }
    doToggleRequest(pendingToggle.el, pendingToggle.id);
    pendingToggle = null;
    confirmModal.hide();
  });

  // perform AJAX toggle (optimistic UI)
  function doToggleRequest(toggleEl, id){
    // optimistic: flip UI
    toggleEl.classList.toggle('active');
    const thumb = toggleEl.querySelector('.ios-thumb');
    if (toggleEl.classList.contains('active')) thumb.style.left = 'calc(var(--track-w) - 28px)'; else thumb.style.left = '4px';

    fetch(location.href, {
      method: 'POST',
      headers: { 'X-Requested-With': 'XMLHttpRequest', 'Content-Type': 'application/x-www-form-urlencoded' },
      body: new URLSearchParams({ toggle: '1', id: id })
    }).then(r=>r.json()).then(json=>{
      if (json && json.ok) {
        const card = document.getElementById('api-card-' + id);
        const nameEl = card.querySelector('.api-name');
        if (json.enabled) {
          // remove inactive label if exists
          nameEl.innerHTML = nameEl.textContent.replace(/\s*inactive\s*$/i,'');
          showToast('API enabled');
        } else {
          if (!/inactive/i.test(nameEl.textContent)) nameEl.innerHTML = nameEl.textContent + '<span style="opacity:.6;font-weight:600;margin-left:6px;font-size:12px;color:#ff7b7b">inactive</span>';
          showToast('API disabled');
        }
      } else {
        // revert UI
        toggleEl.classList.toggle('active');
        if (toggleEl.classList.contains('active')) toggleEl.querySelector('.ios-thumb').style.left='calc(var(--track-w) - 28px)'; else toggleEl.querySelector('.ios-thumb').style.left='4px';
        showToast('Toggle failed', true);
      }
    }).catch(()=> {
      toggleEl.classList.toggle('active');
      if (toggleEl.classList.contains('active')) toggleEl.querySelector('.ios-thumb').style.left='calc(var(--track-w) - 28px)'; else toggleEl.querySelector('.ios-thumb').style.left='4px';
      showToast('Toggle failed', true);
    });
  }

  function showToast(msg, isError){
    document.getElementById('genericToastBody').textContent = msg;
    const t = bootstrap.Toast.getOrCreateInstance(document.getElementById('genericToast'));
    t.show();
  }

  // escape helper
  function escHtml(s){ return String(s).replaceAll('&','&amp;').replaceAll('<','&lt;').replaceAll('>','&gt;'); }

}); // DOMContentLoaded
</script>

<?php include 'partials/footer.php'; ?>
